<?php

/**
 * PublicKeyLoader
 *
 * Returns a PublicKey or PrivateKey object.
 *
 * @category  Crypt
 * @package   PublicKeyLoader
 * @author    Jim Wigginton <terrafrost@php.net>
 * @copyright 2009 Jim Wigginton
 * @license   http://www.opensource.org/licenses/mit-license.html  MIT License
 * @link      http://phpseclib.sourceforge.net
 */
namespace ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt;

use ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\Common\AsymmetricKey;
use ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\Common\PublicKey;
use ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\Common\PrivateKey;
use ZOOlanders\YOOessentials\Vendor\phpseclib3\Exception\NoKeyLoadedException;
use ZOOlanders\YOOessentials\Vendor\phpseclib3\File\X509;
/**
 * PublicKeyLoader
 *
 * @package Common
 * @author  Jim Wigginton <terrafrost@php.net>
 * @access  public
 */
abstract class PublicKeyLoader
{
    /**
     * Loads a public or private key
     *
     * @return AsymmetricKey
     * @access public
     * @param string|array $key
     * @param string $password optional
     */
    public static function load($key, $password = \false)
    {
        try {
            return \ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\EC::load($key, $password);
        } catch (\ZOOlanders\YOOessentials\Vendor\phpseclib3\Exception\NoKeyLoadedException $e) {
        }
        try {
            return \ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\RSA::load($key, $password);
        } catch (\ZOOlanders\YOOessentials\Vendor\phpseclib3\Exception\NoKeyLoadedException $e) {
        }
        try {
            return \ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\DSA::load($key, $password);
        } catch (\ZOOlanders\YOOessentials\Vendor\phpseclib3\Exception\NoKeyLoadedException $e) {
        }
        try {
            $x509 = new \ZOOlanders\YOOessentials\Vendor\phpseclib3\File\X509();
            $x509->loadX509($key);
            $key = $x509->getPublicKey();
            if ($key) {
                return $key;
            }
        } catch (\Exception $e) {
        }
        throw new \ZOOlanders\YOOessentials\Vendor\phpseclib3\Exception\NoKeyLoadedException('Unable to read key');
    }
    /**
     * Loads a private key
     *
     * @return PrivateKey
     * @access public
     * @param string|array $key
     * @param string $password optional
     */
    public function loadPrivateKey($key, $password = \false)
    {
        $key = self::load($key, $password);
        if (!$key instanceof \ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\Common\PrivateKey) {
            throw new \ZOOlanders\YOOessentials\Vendor\phpseclib3\Exception\NoKeyLoadedException('The key that was loaded was not a private key');
        }
        return $key;
    }
    /**
     * Loads a public key
     *
     * @return PublicKey
     * @access public
     * @param string|array $key
     */
    public function loadPublicKey($key)
    {
        $key = self::load($key);
        if (!$key instanceof \ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\Common\PublicKey) {
            throw new \ZOOlanders\YOOessentials\Vendor\phpseclib3\Exception\NoKeyLoadedException('The key that was loaded was not a public key');
        }
        return $key;
    }
    /**
     * Loads parameters
     *
     * @return AsymmetricKey
     * @access public
     * @param string|array $key
     */
    public function loadParameters($key)
    {
        $key = self::load($key);
        if (!$key instanceof \ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\Common\PrivateKey && !$key instanceof \ZOOlanders\YOOessentials\Vendor\phpseclib3\Crypt\Common\PublicKey) {
            throw new \ZOOlanders\YOOessentials\Vendor\phpseclib3\Exception\NoKeyLoadedException('The key that was loaded was not a parameter');
        }
        return $key;
    }
}
