<?php

/*
 * This file is part of the league/commonmark package.
 *
 * (c) Colin O'Dell <colinodell@gmail.com>
 *
 * Original code based on the CommonMark JS reference parser (https://bitly.com/commonmark-js)
 *  - (c) John MacFarlane
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Parser;

use ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListBlock;
use ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListData;
use ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListItem;
use ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\Paragraph;
use ZOOlanders\YOOessentials\Vendor\League\CommonMark\ContextInterface;
use ZOOlanders\YOOessentials\Vendor\League\CommonMark\Cursor;
use ZOOlanders\YOOessentials\Vendor\League\CommonMark\Util\ConfigurationAwareInterface;
use ZOOlanders\YOOessentials\Vendor\League\CommonMark\Util\ConfigurationInterface;
use ZOOlanders\YOOessentials\Vendor\League\CommonMark\Util\RegexHelper;
final class ListParser implements \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Parser\BlockParserInterface, \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Util\ConfigurationAwareInterface
{
    /** @var ConfigurationInterface|null */
    private $config;
    /** @var string|null */
    private $listMarkerRegex;
    public function setConfiguration(\ZOOlanders\YOOessentials\Vendor\League\CommonMark\Util\ConfigurationInterface $configuration)
    {
        $this->config = $configuration;
    }
    public function parse(\ZOOlanders\YOOessentials\Vendor\League\CommonMark\ContextInterface $context, \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Cursor $cursor) : bool
    {
        if ($cursor->isIndented() && !$context->getContainer() instanceof \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListBlock) {
            return \false;
        }
        $indent = $cursor->getIndent();
        if ($indent >= 4) {
            return \false;
        }
        $tmpCursor = clone $cursor;
        $tmpCursor->advanceToNextNonSpaceOrTab();
        $rest = $tmpCursor->getRemainder();
        if (\preg_match($this->listMarkerRegex ?? $this->generateListMarkerRegex(), $rest) === 1) {
            $data = new \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListData();
            $data->markerOffset = $indent;
            $data->type = \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListBlock::TYPE_BULLET;
            $data->delimiter = null;
            $data->bulletChar = $rest[0];
            $markerLength = 1;
        } elseif (($matches = \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Util\RegexHelper::matchAll('/^(\\d{1,9})([.)])/', $rest)) && (!$context->getContainer() instanceof \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\Paragraph || $matches[1] === '1')) {
            $data = new \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListData();
            $data->markerOffset = $indent;
            $data->type = \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListBlock::TYPE_ORDERED;
            $data->start = (int) $matches[1];
            $data->delimiter = $matches[2];
            $data->bulletChar = null;
            $markerLength = \strlen($matches[0]);
        } else {
            return \false;
        }
        // Make sure we have spaces after
        $nextChar = $tmpCursor->peek($markerLength);
        if (!($nextChar === null || $nextChar === "\t" || $nextChar === ' ')) {
            return \false;
        }
        // If it interrupts paragraph, make sure first line isn't blank
        $container = $context->getContainer();
        if ($container instanceof \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\Paragraph && !\ZOOlanders\YOOessentials\Vendor\League\CommonMark\Util\RegexHelper::matchAt(\ZOOlanders\YOOessentials\Vendor\League\CommonMark\Util\RegexHelper::REGEX_NON_SPACE, $rest, $markerLength)) {
            return \false;
        }
        // We've got a match! Advance offset and calculate padding
        $cursor->advanceToNextNonSpaceOrTab();
        // to start of marker
        $cursor->advanceBy($markerLength, \true);
        // to end of marker
        $data->padding = $this->calculateListMarkerPadding($cursor, $markerLength);
        // add the list if needed
        if (!$container instanceof \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListBlock || !$data->equals($container->getListData())) {
            $context->addBlock(new \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListBlock($data));
        }
        // add the list item
        $context->addBlock(new \ZOOlanders\YOOessentials\Vendor\League\CommonMark\Block\Element\ListItem($data));
        return \true;
    }
    /**
     * @param Cursor $cursor
     * @param int    $markerLength
     *
     * @return int
     */
    private function calculateListMarkerPadding(\ZOOlanders\YOOessentials\Vendor\League\CommonMark\Cursor $cursor, int $markerLength) : int
    {
        $start = $cursor->saveState();
        $spacesStartCol = $cursor->getColumn();
        while ($cursor->getColumn() - $spacesStartCol < 5) {
            if (!$cursor->advanceBySpaceOrTab()) {
                break;
            }
        }
        $blankItem = $cursor->peek() === null;
        $spacesAfterMarker = $cursor->getColumn() - $spacesStartCol;
        if ($spacesAfterMarker >= 5 || $spacesAfterMarker < 1 || $blankItem) {
            $cursor->restoreState($start);
            $cursor->advanceBySpaceOrTab();
            return $markerLength + 1;
        }
        return $markerLength + $spacesAfterMarker;
    }
    private function generateListMarkerRegex() : string
    {
        // No configuration given - use the defaults
        if ($this->config === null) {
            return $this->listMarkerRegex = '/^[*+-]/';
        }
        $markers = $this->config->get('unordered_list_markers', ['*', '+', '-']);
        if (!\is_array($markers)) {
            throw new \RuntimeException('Invalid configuration option "unordered_list_markers": value must be an array of strings');
        }
        return $this->listMarkerRegex = '/^[' . \preg_quote(\implode('', $markers), '/') . ']/';
    }
}
