<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Source;

use function YOOtheme\app;
use GraphQL\Type\Definition\ObjectType;
use YOOtheme\Builder\Source;
use YOOtheme\Config;
use YOOtheme\Str;
use ZOOlanders\YOOessentials\Addon;

class SourceService
{
    public const SOURCES_CONFIG_KEY = '~theme.yooessentials.source.sources';

    /** @var bool */
    protected $enabled = true;

    /** @var array */
    protected $sources = [];

    /** @var array */
    protected $providers = [];

    public function __construct(Config $config)
    {
        if (!Addon::enabled(self::class)) {
            return;
        }

        $this->providers = [];
        $this->sources = $config->get(self::SOURCES_CONFIG_KEY, []);
    }

    public function setSources(array $sources): self
    {
        $this->sources = $sources;

        return $this;
    }

    /**
     * @param string|SourceProviderInterface $sourceProvider
     * @return $this
     */
    public function addSourceProvider($provider): self
    {
        if (!isset($this->providers[$provider])) {
            $provider = is_string($provider) ? app($provider) : $provider;
            $this->providers[$provider->config()->name] = $provider;
        }

        return $this;
    }

    /**
     * @return array[]|array
     */
    public function sources(?string $name = null): array
    {
        if (!$name) {
            return $this->sources;
        }

        return array_filter($this->sources, function ($source) use ($name) {
            return ($source['provider'] ?? '') === $name;
        });
    }

    /**
     * @return SourceProviderInterface[]|array
     */
    public function providers(): array
    {
        return $this->providers;
    }

    public function provider(string $name): ?SourceProviderInterface
    {
        return $this->providers[$name] ?? null;
    }

    public function setObjectType(Source $source, string $name, array $config): void
    {
        $type = new ObjectType([
            'name' => $name, 'resolveField' => [$source, 'resolveField'],
        ]);

        $type->config = $config;
        $source->setType($type);
    }

    public static function encodeField(string $field): string
    {
        return Str::lower(preg_replace('/%.{2}/', '_', rawurlencode($field)));
    }
}
