<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Source;

use function YOOtheme\app;
use YOOtheme\Builder\Source;
use YOOtheme\Config;
use YOOtheme\Http\Request;
use ZOOlanders\YOOessentials\Addon;
use ZOOlanders\YOOessentials\Source\Providers\Instagram\InstagramApi;

class SourcePreviewListener
{
    /**
     * @param Config $config
     * @param Request $request
     * @param SourceService $service
     * @param Source $source
     * @return false|void
     */
    public static function initSource(Config $config, Request $request, SourceService $service, $source)
    {
        if (!Addon::enabled(SourceService::class)) {
            return;
        }

        $wpPreview = self::initSourceWordpress($request, $config, $source, $service);
        if ($wpPreview === false) {
            return false;
        }

        return self::initSourceJoomla($request, $config, $source, $service);
    }

    protected static function initSourceJoomla(Request $request, Config $config, Source $source, SourceService $service)
    {
        $previewOauths = json_decode($request->getParam('yooessentialsOAuth'), true) ?? [];
        $previewSources = json_decode($request->getParam('yooessentialsSources'), true) ?? [];

        if (!$previewSources) {
            return;
        }

        // init core source, set customizer attribute to skip using cached schema
        $request->setAttribute('customizer', true);
        $result = Source\SourceListener::initSource($config, $request, $source);

        // add our sources
        self::addPreviewSources($config, $service, $source, $previewSources, $previewOauths);

        return $result;
    }

    protected static function initSourceWordpress(Request $request, Config $config, Source $source, SourceService $service)
    {
        $page = $request->getParam('customized');
        if (!$page) {
            return;
        }

        // It's json encoded "twice" with backslashed, so fake i as a json string
        $page = json_decode('"' . $page . '"', true);
        if (!$page) {
            return;
        }

        $page = json_decode($page, true);
        if (!$page) {
            return;
        }

        $page = $page['page'] ?? null;
        if (!$page) {
            return;
        }

        $page = base64_decode($page);
        if (!$page) {
            return;
        }

        $page = json_decode($page, true);
        if (!$page) {
            return;
        }

        $previewOauths = $page['yooessentialsOAuth'] ?? null;
        $previewSources = $page['yooessentialsSources'] ?? null;

        if (!$previewSources) {
            return;
        }

        // init core source, set customizer attribute to skip using cached schema
        $request->setAttribute('customizer', true);
        $result = Source\SourceListener::initSource($config, $request, $source);

        // add our sources
        self::addPreviewSources($config, $service, $source, $previewSources, $previewOauths);

        return $result;
    }

    protected static function addPreviewSources(Config $config, SourceService $service, Source $source, array $previewSources, array $previewOauths): void
    {
        $service->setSources($previewSources);

        if ($previewOauths && count($previewOauths) > 0) {
            $config->set(ConnectedAccountsService::ACCOUNTS_CONFIG_KEY, $previewOauths);
            $accountService = app(ConnectedAccountsService::class);
            $accountService->setAccounts($accountService->loadAccounts($config));

            $instagramApi = app(InstagramApi::class);
            foreach ($previewOauths as $token) {
                $instagramApi->addToken(new ConnectedAccount($token));
            }
        }

        foreach ($service->providers() as $provider) {
            if ($provider && $provider instanceof SourceProviderInterface) {
                $name = $provider->config()->name;
                $sourcesConfig = array_filter($previewSources, function ($s) use ($name) {
                    return ($s['provider'] ?? '') === $name;
                });
                $provider->initSource($source, $sourcesConfig);
            }
        }
    }
}
