<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Source\Providers\Instagram;

use DateTime;
use Exception;
use function YOOtheme\app;
use YOOtheme\HttpClientInterface;
use ZOOlanders\YOOessentials\ConfigurationRepository;
use ZOOlanders\YOOessentials\Source\ConnectedAccount;
use ZOOlanders\YOOessentials\Source\ConnectedAccountsService;
use ZOOlanders\YOOessentials\Vendor\Symfony\Contracts\Cache\CacheInterface;
use ZOOlanders\YOOessentials\Vendor\Symfony\Contracts\Cache\ItemInterface;

class InstagramApi
{
    /** @var ConnectedAccount[] */
    protected $tokens = [];

    /** @var ConfigurationRepository */
    protected $repository;

    /** @var HttpClientInterface */
    protected $httpClient;

    /** @var ConnectedAccountsService */
    private $accountsService;

    public function __construct(ConfigurationRepository $repository, HttpClientInterface $httpClient, ConnectedAccountsService $accountsService)
    {
        $this->repository = $repository;
        $this->httpClient = $httpClient;
        $this->accountsService = $accountsService;

        // Take only the instagram / facebook tokens
        $this->tokens = array_filter($this->accountsService->accounts(), function (ConnectedAccount $account) {
            return in_array($account->driver, [
                'instagrambasic',
                'instagram',
                'facebook'
            ]);
        });

        $this->refreshTokens();
    }

    public function personalAccounts(): array
    {
        return array_values(array_filter($this->tokens, function (ConnectedAccount $token) {
            return stripos($token->driver, 'instagram') === 0;
        }));
    }

    public function addToken(ConnectedAccount $token): self
    {
        $this->tokens[] = $token;

        return $this;
    }

    public function pages(array $accountIds = [], bool $force = false): array
    {
        /** @var CacheInterface $cache */
        $cache = app(CacheInterface::class);

        $tokens = $this->tokens;
        if (count($accountIds) > 0) {
            $tokens = array_filter($tokens, function (ConnectedAccount $token) use ($accountIds) {
                return in_array($token->id, $accountIds);
            });
        }

        $tokenIds = array_map(function (ConnectedAccount $account) {
            return $account->id;
        }, $tokens);

        $cacheKey = 'instagram-pages-' . sha1(json_encode($tokenIds));

        if ($force) {
            $cache->delete($cacheKey);
        }

        return $cache->get($cacheKey, function (ItemInterface $item) use ($tokens) {
            $item->expiresAfter(3600);

            $pages = [];
            foreach ($tokens as $token) {
                $userId = $token->id;
                $pages[$userId] = [];

                $pageIds = $this->userPageIds($userId);

                foreach ($pageIds as $pageId) {
                    $pages[$userId][$pageId] = $this->instagramPageDetails($userId, $pageId);
                }
                $pages[$userId] = array_values($pages[$userId]);
            }

            return $pages;
        });
    }

    public function isConfigured(): bool
    {
        return count($this->tokens) > 0;
    }

    public function userPageIds(string $userId): array
    {
        $config = $this->configForUser($userId);
        $token = $config->accessToken;
        $query = 'https://graph.facebook.com/me/accounts?access_token=%s';

        $url = sprintf($query, $token);
        $result = $this->httpClient->get($url);

        $data = json_decode((string) $result->getBody(), true)['data'] ?? [];

        return array_filter(array_map(function ($page) use ($userId) {
            return $this->userInstagramPageId($userId, $page['id'] ?? '');
        }, $data));
    }

    public function instagramPageDetails(string $userId, string $pageId): array
    {
        $config = $this->configForUser($userId);
        $token = $config->accessToken;
        $query = 'https://graph.facebook.com/%s?fields=name&access_token=%s';

        $url = sprintf($query, $pageId, $token);
        $result = $this->httpClient->get($url);

        $data = json_decode((string) $result->getBody(), true);

        return [
            'id' => $pageId,
            'userId' => $userId,
            'name' => $data['name'] ?? $pageId
        ];
    }

    public function userInstagramPageId(string $userId, string $fbPageId): ?string
    {
        $config = $this->configForUser($userId);
        $token = $config->accessToken;
        $query = 'https://graph.facebook.com/%s?fields=instagram_business_account&access_token=%s';

        $url = sprintf($query, $fbPageId, $token);
        $result = $this->httpClient->get($url);

        $data = json_decode((string) $result->getBody(), true);

        if (!$data) {
            return null;
        }

        $instagram = $data['instagram_business_account'] ?? null;
        if (!$instagram) {
            return null;
        }

        return $instagram['id'];
    }

    public function mediaForUser(string $userId, int $limit = 20, bool $shouldTryToRefresh = true): array
    {
        $config = $this->configForUser($userId);
        if (!$config) {
            return [];
        }

        $token = $config->accessToken;
        $url = "https://graph.instagram.com/me/media?fields=id,caption,media_type,media_url,owner,permalink,thumbnail_url,timestamp,children{media_type,media_url,permalink,thumbnail_url}&access_token={$token}&limit={$limit}";

        $result = $this->httpClient->get($url);

        $body = json_decode((string) $result->getBody(), true);
        $error = $body['error'] ?? [];
        if (!$error) {
            return $body['data'] ?? [];
        }

        if ($shouldTryToRefresh && ($error['type'] ?? null) === 'OAuthException') {
            $this->refreshPersonalToken($config);

            return $this->mediaForUser($userId, $limit, false);
        }

        throw new Exception($error['message'] ?? 'Generic instagram error', $error['code'] ?? 400);
    }

    public function mediaForPage(string $pageId, string $userId, int $limit = 20): array
    {
        $config = $this->configForUser($userId);
        if (!$config) {
            return [];
        }

        $token = $config->accessToken;
        $url = "https://graph.facebook.com/{$pageId}/media?fields=media_url,thumbnail_url,caption,id,media_type,timestamp,username,comments_count,like_count,permalink,children{media_url,id,media_type,timestamp,permalink,thumbnail_url}&access_token={$token}&limit={$limit}";

        $result = $this->httpClient->get($url);

        return json_decode((string) $result->getBody(), true)['data'] ?? [];
    }

    public function configForUser(string $userId): ?ConnectedAccount
    {
        foreach ($this->tokens as $token) {
            if ($token->id === $userId) {
                return $token;
            }
        }

        return null;
    }

    public function refreshTokens(): void
    {
        $atLeastOneTokenRefreshed = false;
        /** @var ConnectedAccount $token */
        foreach ($this->tokens as &$token) {
            if (!$token->isTokenExpiring()) {
                continue;
            }

            $atLeastOneTokenRefreshed = true;

            switch ($token->driver) {
                case 'facebook':
                    $token = $this->refreshFacebookToken($token);

                    break;
                case 'instagram':
                case 'instagrambasic':
                default:
                    $token = $this->refreshPersonalToken($token);

                    break;
            }
        }

        if (!$atLeastOneTokenRefreshed) {
            return;
        }

        $this->accountsService->saveAccounts($this->tokens);
    }

    protected function refreshPersonalToken(ConnectedAccount $token): ConnectedAccount
    {
        $url = "https://graph.instagram.com/refresh_access_token?grant_type=ig_refresh_token&access_token={$token->accessToken}";

        try {
            $client = app(HttpClientInterface::class);
            $result = $client->get($url);
        } catch (\Exception $e) {
            return $token->accessToken;
        }

        $data = json_decode((string) $result->getBody(), true);

        $token->accessToken = $data['access_token'] ?? $token->accessToken;

        $expiresAt = null;
        $expiresIn = $data['expires'] ?? (((int) (new DateTime)->format('U'))) + ($data['expires_in'] ?? $data['expiresIn'] ?? (60 * 24 * 60 * 60));

        if ($expiresIn !== null) {
            $now = (int) (new DateTime())->format('U');
            $expiresAt = new DateTime();
            $expiresAt->setTimestamp($now + $expiresIn);
        }

        $token->expiresIn = $expiresIn;
        $token->expiresAt = $expiresAt;

        return $token;
    }

    protected function refreshFacebookToken(ConnectedAccount $token): ConnectedAccount
    {
        $query = 'https://instagram.zoolanders.com?user_id=%s&facebookAccessToken=%s';

        $url = sprintf($query, $token->id, $token->accessToken);

        try {
            $client = app(HttpClientInterface::class);
            $result = $client->get($url);
        } catch (\Exception $e) {
            return $token;
        }

        $data = json_decode((string) $result->getBody(), true);

        if (!$data || !$data['access_token']) {
            return $token;
        }

        $token->accessToken = $data['access_token'] ?? $token->accessToken;
        $token->id = $data['user_id'] ?? $token->id;
        $token->username = $data['username'] ?? $token->username;

        $expiresAt = null;
        $expiresIn = $data['expiresIn'] ?? null;

        if ($expiresIn !== null) {
            $now = (int) (new DateTime())->format('U');
            $expiresAt = new DateTime();
            $expiresAt->setTimestamp($now + $expiresIn);
        }

        $token->expiresIn = $expiresIn;
        $token->expiresAt = $expiresAt;

        return $token;
    }
}
