<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Source\Providers\GoogleSheet\Type;

use function YOOtheme\app;
use ZOOlanders\YOOessentials\Source\Providers\GoogleSheet\HasGoogleClient;
use ZOOlanders\YOOessentials\Source\Providers\GoogleSheet\Provider;
use ZOOlanders\YOOessentials\Source\SourceService;
use ZOOlanders\YOOessentials\Vendor\Symfony\Component\Cache\Adapter\FilesystemAdapter;
use ZOOlanders\YOOessentials\Vendor\Symfony\Contracts\Cache\CacheInterface;
use ZOOlanders\YOOessentials\Vendor\Symfony\Contracts\Cache\ItemInterface;

class SheetQueryType
{
    use HasGoogleClient;

    public static function config(string $name, string $label, string $type, string $sheetId): array
    {
        return [

            'fields' => [

                $name => [

                    'type' => ['listOf' => $type],

                    'args' => [
                        'sheet_id' => [
                            'type' => 'ID',
                        ],
                        'start_column' => [
                            'type' => 'String',
                        ],
                        'end_column' => [
                            'type' => 'String',
                        ],
                        'cache' => [
                            'type' => 'Int'
                        ],
                        'offset' => [
                            'type' => 'Int',
                        ],
                        'limit' => [
                            'type' => 'Int',
                        ],

                    ],

                    'metadata' => [
                        'group' => 'Google Sheet',
                        'label' => $label,
                        'fields' => [

                            'start_column' => [
                                'label' => 'First Column',
                                'description' => 'Start from this column name (defaults to A)',
                                'type' => 'text',
                                'default' => 'A'
                            ],

                            'end_column' => [
                                'label' => 'Last Column',
                                'description' => 'End on this column (defaults to Z)',
                                'type' => 'text',
                                'default' => 'Z'
                            ],

                            'offset' => [
                                'label' => 'Offset',
                                'description' => 'How many rows should we skip (excluding the header)?',
                                'type' => 'number',
                                'default' => 0
                            ],

                            'limit' => [
                                'label' => 'Limit',
                                'description' => 'How many rows in total to fetch?',
                                'type' => 'number',
                                'default' => 20
                            ],

                            'cache' => [
                                'type' => 'number',
                                'label' => 'Cache Time (secs)',
                                'description' => 'Cache duration. Defaults to 1 hour',
                                'default' => 3600
                            ],
                        ],
                    ],

                    'extensions' => [
                        'call' => [
                            'func' => __CLASS__ . '::resolve',
                            'args' => [
                                'sheet_id' => $sheetId,
                            ]
                        ]
                    ],

                ],

            ],

        ];
    }

    public static function resolve($item, $args)
    {
        if (!$sheetId = $args['sheet_id'] ?? null) {
            return [];
        }

        /** @var FilesystemAdapter $cache */
        $cache = app(CacheInterface::class);
        $cacheKey = 'google-sheet-' . sha1(json_encode($args));

        return $cache->get($cacheKey, function (ItemInterface $item) use ($args, $sheetId) {
            $item->expiresAfter($args['cache'] ?? 3600);

            if (
                !$accountId = Provider::googleAccount($sheetId) or
                !$client = Provider::googleClient($accountId) or
                !$spreadSheet = Provider::googleSpreadSheet($client, $sheetId)
            ) {
                return [];
            }

            $headers = Provider::getHeaders($args, $spreadSheet, $client);
            $values = Provider::getValues($args, $spreadSheet, $client);

            $data = [];
            foreach ($values as $row) {
                $rowData = [];
                foreach ($row as $k => $value) {
                    $key = $headers[$k];
                    $rowData[SourceService::encodeField($key)] = $value;
                }

                $data[] = $rowData;
            }

            return $data;
        });
    }
}
