<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Source\Providers\Csv;

use function YOOtheme\app;
use YOOtheme\Builder\Source;
use YOOtheme\Event;
use YOOtheme\File;
use YOOtheme\Path;
use YOOtheme\Str;
use ZOOlanders\YOOessentials\Source\SourceProviderInterface;
use ZOOlanders\YOOessentials\Source\SourceService;
use ZOOlanders\YOOessentials\Util\Prop;
use ZOOlanders\YOOessentials\Vendor\League\Csv\Reader;

class Provider implements SourceProviderInterface
{
    public function config(): object
    {
        return (object) app()->config->loadFile(Path::get('./provider.json'));
    }

    public function initSource(Source $source, array $config = []): void
    {
        /** @var SourceService $service */
        $service = app(SourceService::class);

        foreach ($config as $args) {
            $file = self::getFilePath($args);

            if (!$file) {
                continue;
            }

            try {
                $csv = self::readCSV($args);
                $name = self::getQueryName($args);
                $label = self::getTypeLabel($args);
                $type = self::getTypeName($csv);

                $source->queryType(Type\CsvQueryType::config($name, $label, $type, $csv));
                $service->setObjectType($source, $type, Type\CsvFileType::config($csv, $label));
            } catch (\Exception $e) {
                Event::emit('yooessentials.error', [
                    'addon' => 'source',
                    'provider' => 'csv',
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    public static function getFilePath(array $args): ?string
    {
        $file = $args['file'] ?? null;

        if ($file and !Str::startsWith($file, '~')) {
            $file = "~/$file";
        }

        if (!File::exists($file)) {
            return null;
        }

        return $file;
    }

    public static function getQueryName(array $args): string
    {
        $id = self::getQueryId($args);

        return "fileCSV_{$id}_query";
    }

    public static function getQueryId(array $args): string
    {
        $createdOn = $args['_meta']['created_on'] ?? '';

        // legacy
        if (version_compare($createdOn, '1.2.4', '<=')) {
            return sha1(self::getFilePath($args));
        }

        return $args['id'];
    }

    public static function getTypeName(Reader $csv): string
    {
        return 'fileCSV_' . sha1(json_encode(array_filter($csv->getHeader())));
    }

    public static function getTypeLabel(array $args): string
    {
        return $args['name'] ?? "CSV - {$args['id']}";
    }

    public static function readCSV(array $args): Reader
    {
        $file = self::getFilePath($args);
        $enclosure = Prop::parseString($args, 'enclosure', '"', 1);
        $delimiter = Prop::parseString($args, 'delimiter', ',', 1);

        return Reader::createFromPath(File::get($file), 'r')
            ->setHeaderOffset(0)
            ->setEnclosure($enclosure)
            ->setDelimiter($delimiter);
    }
}
