<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Form\Action;

use function YOOtheme\app;
use RuntimeException;
use stdClass;
use YOOtheme\File;
use YOOtheme\Path;
use YOOtheme\Str;
use ZOOlanders\YOOessentials\Form\Http\FormSubmissionResponse;
use ZOOlanders\YOOessentials\Util\Prop;

class SaveCsvAction extends StandardAction implements Action
{
    protected $configKey = 'action-save-csv';

    public const DOWNLOAD_CSV_URL = '/yooessentials/form-download-csv';

    public function name() : string
    {
        return 'Save CSV';
    }

    public function panel(): array
    {
        return app()->config->loadFile(Path::get('../../config/action/savecsv.json'));
    }

    public function __invoke(FormSubmissionResponse $response, callable $next): FormSubmissionResponse
    {
        $form = $response->submission()->form();
        $config = (object) $this->getConfig();

        if (!Str::startsWith($config->path, '~')) {
            $config->path = "~/$config->path";
        }

        if (!File::exists($config->path) && !File::makeDir($config->path, 0755, true)) {
            throw new RuntimeException('Cannot create directory ' . $config->path);
        }

        $config->enclosure = Prop::parseString($config, 'enclosure', '"', 1);
        $config->separator = Prop::parseString($config, 'separator', ',', 1);

        $controls = $form->controls();
        $columnsConfig = self::columnsConfig($config, $controls);
        $formData = self::flattenFormData($response->submission()->data());

        // map formData with columns order and configuration
        $data = array_reduce($columnsConfig['fields'], function ($carry, $field) use ($formData) {
            return array_merge($carry, [$formData[$field]]);
        }, []);

        $fileid = hash('crc32b', json_encode($columnsConfig['headers']));
        $filename = trim(basename($config->file ?? $form->id(), '.csv'), ' /');
        $filepath = Path::resolve($config->path, "$filename-$fileid.csv");

        self::writeHeader($filepath, $columnsConfig['headers'], $config->separator, $config->enclosure);

        $handle = fopen($filepath, 'a');
        fputcsv($handle, $data, $config->separator, $config->enclosure);
        fclose($handle);

        return $next($response);
    }

    protected static function columnsConfig(stdClass $config, array $controls): array
    {
        $columns = !empty($config->columns)
            ? $config->columns
            : array_map(function ($control) {
                return $control['name'];
            }, $controls);

        $columns = array_filter($columns);

        return array_reduce($columns, function ($carry, $col) {
            $carry['headers'][] = $col['title'] ?? $col;
            $carry['fields'][] = $col['field'] ?? $col;

            return $carry;
        }, ['headers' => [], 'fields' => []]);
    }

    protected static function flattenFormData(array $formData, string $rootKey = ''): array
    {
        $data = [];

        array_walk($formData, function ($value, $key) use (&$data, $rootKey) {
            $newKey = $key;
            if ($rootKey) {
                $newKey = $rootKey . ' ' . $newKey;
            }

            if (is_array($value)) {
                $data[$newKey] = implode(', ', self::flattenFormData($value));

                return;
            }

            $data[$newKey] = $value;
        });

        return $data;
    }

    protected static function writeHeader(string $fullFileName, array $header, string $separator, string $enclosure): void
    {
        $handle = fopen($fullFileName, 'a+');
        $oldHeader = fgetcsv($handle);

        // empty file, just write header
        if (!$oldHeader) {
            fputcsv($handle, $header, $separator, $enclosure);
        }

        fclose($handle);
    }
}
