<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Form\Action;

use function YOOtheme\app;
use YOOtheme\Arr;
use YOOtheme\Path;
use ZOOlanders\YOOessentials\Form\Http\FormSubmissionResponse;
use ZOOlanders\YOOessentials\Mailer;

class EmailAction extends StandardAction implements Action
{
    public const TEST_EMAIL_URL = '/yooessentials/form-send-test-email';

    protected $configKey = 'action-email';

    public function name(): string
    {
        return 'Email';
    }

    public function panel(): array
    {
        return app()->config->loadFile(Path::get('../../config/action/email.json'));
    }

    public function __invoke(FormSubmissionResponse $response, callable $next): FormSubmissionResponse
    {
        $config = self::prepareConfig($this->getConfig(), $response);

        if (count($config['recipients']) <= 0) {
            return $next($response);
        }

        $mailer = self::prepareMailer($config);

        try {
            $mailer->send();
        } catch (\Exception $e) {
            $response->withErrors([
                "Email Action Error: {$e->getMessage()}"
            ]);
        }

        return $next($response);
    }

    public static function prepareConfig(array $config, FormSubmissionResponse $response): array
    {
        // replace tags
        foreach (['subject', 'body', 'recipients', 'ccs', 'bccs', 'reply_tos', 'from'] as $emailField) {
            if (!($config[$emailField] ?? false)) {
                continue;
            }

            $config[$emailField] = $response->submission()->parseTags($config[$emailField]);
        }

        $config['from'] = $config['from'] ?? '';
        $config['body'] = $config['body'] ?? '';
        $config['html'] = $config['html'] ?? false;
        $config['subject'] = $config['subject'] ?? '';

        $config['ccs'] = self::explode($config['ccs'] ?? '');
        $config['bccs'] = self::explode($config['bccs'] ?? '');
        $config['reply_tos'] = self::explode($config['reply_tos'] ?? '');
        $config['recipients'] = self::explode($config['recipients'] ?? '');

        $config['attachments'] = array_filter($config['attachments'] ?? []);
        $config['submitted_attachments'] = array_filter($config['submitted_attachments'] ?? []);

        // merge attachments
        foreach ($config['submitted_attachments'] ?? [] as $field) {
            $attachments = Arr::wrap($response->submission()->data($field) ?: []);
            $config['attachments'] = array_merge($config['attachments'], $attachments);
        }

        unset($config['submitted_attachments']);

        return $config;
    }

    public static function prepareMailer(array $config): Mailer
    {
        /** @var Mailer $mailer */
        $mailer = app(Mailer::class);

        foreach ($config['recipients'] as $recipient) {
            $mailer->addRecipient($recipient);
        }

        foreach ($config['ccs'] as $cc) {
            $mailer->addCc($cc);
        }

        foreach ($config['bccs'] as $bcc) {
            $mailer->addBcc($bcc);
        }

        foreach ($config['reply_tos'] as $replyTo) {
            $mailer->addReplyTo($replyTo);
        }

        foreach ($config['attachments'] as $attachment) {
            $mailer->addAttachment($attachment);
        }

        if ($from = $config['from']) {
            $mailer->setFrom($from);
        }

        if ($config['subject']) {
            $mailer->setSubject($config['subject']);
        }

        if ($config['body']) {
            $mailer->setBody($config['body']);
        }

        if ($config['html']) {
            $mailer->isHtml(true);
        }

        return $mailer;
    }

    protected static function explode(string $string): array
    {
        return array_filter(array_map('trim', explode(',', $string)));
    }
}
