<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Form;

use function YOOtheme\App;
use YOOtheme\Arr;
use YOOtheme\Config;
use YOOtheme\File;
use YOOtheme\Http\Message\UploadedFile;
use YOOtheme\Path;
use YOOtheme\Str;
use ZOOlanders\YOOessentials\Form\Http\FormSubmissionRequest;
use ZOOlanders\YOOessentials\Form\Http\FormSubmissionResponse;
use ZOOlanders\YOOessentials\Form\Validation\Size;
use ZOOlanders\YOOessentials\Util;
use ZOOlanders\YOOessentials\Vendor\Respect\Validation\Exceptions\ContainsAnyException;
use ZOOlanders\YOOessentials\Vendor\Respect\Validation\Exceptions\RegexException;
use ZOOlanders\YOOessentials\Vendor\Respect\Validation\Exceptions\ValidationException;
use ZOOlanders\YOOessentials\Vendor\Respect\Validation\Factory;
use ZOOlanders\YOOessentials\Vendor\Respect\Validation\Rules\Extension;
use ZOOlanders\YOOessentials\Vendor\Respect\Validation\Rules\Mimetype;
use ZOOlanders\YOOessentials\Vendor\Respect\Validation\Validator;

return [

    'transforms' => [

        'render' => function ($node) {

            /** @var FormSubmissionRequest $submission */
            $submission = app(FormSubmissionRequest::class);

            $controlName = $node->controls->field['name'];

            $node->control = (object) [
                'name' => $controlName,
                'errors' => $submission->validator()->errors($controlName) ?? [],
                'value' => $submission->data($controlName),
                'props' => $node->controls->field['props']
            ];

            $node->button = Util\Prop::filterByPrefix($node->props, 'button_');
        }

    ],

    'controls' => [

        'field' => function ($node) {
            $props = Util\Prop::filterByPrefix($node->props, 'control_');
            $name = $props['name'] ?: "upload-$node->id";

            return compact('name', 'props');
        }

    ],

    'validation' => function ($control, Validator $validator, FormSubmissionRequest $submission) {
        $props = $control['props'];
        $files = $submission->request()->getUploadedFile($control['name']);

        $mimeTypes = $props['mimetypes'] ?? '';
        $mimeTypes = array_filter(explode(',', str_replace(' ', '', $mimeTypes)));

        $extensions = $props['extensions'] ?? '';
        $extensions = array_filter(explode(',', str_replace([' ', '.'], '', $extensions)));

        /** @var UploadedFile $file */
        foreach (Arr::wrap($files) as $file) {
            if ($file->getError()) {
                continue;
            }

            try {
                if (count($mimeTypes) > 0) {
                    try {
                        $v = clone $validator;

                        $validations = array_map(function ($type) {
                            return Validator::regex('#' . str_replace('*', '[a-z]*', $type) . '#');
                        }, $mimeTypes);

                        $v->anyOf(...$validations)->check($file->getClientMediaType());
                    } catch (RegexException $e) {
                        throw Factory::getDefaultInstance()->exception((new Mimetype(implode(',', $mimeTypes))), $e->getId());
                    }
                }

                if (count($extensions) > 0) {
                    try {
                        $v = clone $validator;
                        $v->containsAny($extensions)->check($file->getClientFilename());
                    } catch (ContainsAnyException $e) {
                        throw Factory::getDefaultInstance()->exception((new Extension(implode(',', $extensions))), $e->getId());
                    }
                }

                $minSize = $props['min_filesize'] ?? null;
                $maxSize = $props['max_filesize'] ?? null;

                if ($minSize || $maxSize) {
                    try {
                        (new Size($minSize, $maxSize))->check($file);
                    } catch (ValidationException $e) {
                        throw Factory::getDefaultInstance()->exception(new Size($minSize, $maxSize), $validator->getName());
                    }
                }
            } catch (ValidationException $e) {
                throw $e;
            }
        }

        if ($props['required'] ?? false) {
            $v = clone $validator;
            $v->notOptional();

            foreach (Arr::wrap($files) as $file) {
                $v->notOptional()->check($file->getClientFilename());
            }
        }

        return $validator;
    },

    'submission' => function ($control, FormSubmissionRequest $submission, FormSubmissionResponse $response) {
        $props = $control['props'];
        $path = $props['upload_path'];
        $controlName = $control['name'];
        $errorTmpl = "File Submission failed due to server error: '%s'";

        /** @var Config $config */
        $config = app(Config::class);

        if (!Str::startsWith($path, '~')) {
            $path = "~/$path";
        }

        // when childtheme base path base must be resolved independently
        if (Str::startsWith($path, '~theme') && $childDir = $config->get('theme.childDir')) {
            $path = Path::join($childDir, str_replace('~theme', '', $path));
        } else {
            $path = Path::resolve($path);
        }

        // resolve path tags
        $path = $submission->parseTags($path);

        if (!File::makeDir($path, 0777, true)) {
            $response->withErrors([
                sprintf($errorTmpl, 'Failed to create destination folder.')
            ]);

            return;
        }

        $files = Arr::wrap($submission->request()->getUploadedFile($controlName));

        /** @var UploadedFile $file */
        foreach ($files as $i => $file) {
            if (!$file instanceof UploadedFile || $file->getError()) {
                continue;
            }

            $destination = $path . '/' . $file->getClientFilename();
            $destination = Util\File::getUniqueFilepath($destination);

            try {
                $file->moveTo($destination);

                // Override submission data
                $data = $submission->data();
                if (isset($data[$controlName][$i])) {
                    $data[$controlName][$i] = $destination;
                } else {
                    $data[$controlName] = $destination;
                }

                $submission->setData($data);
            } catch (\Exception $e) {
                $response->withErrors([
                    sprintf($errorTmpl, $e->getMessage())
                ]);
            }
        }
    },

];
