<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Wordpress;

use YOOtheme\File;
use YOOtheme\Path;
use YOOtheme\Str;
use ZOOlanders\YOOessentials\Mailer as MailerInterface;

class Mailer implements MailerInterface
{
    protected $headers = [];
    protected $to = [];
    protected $body = '';
    protected $subject = '';
    protected $html = false;
    protected $attachments = [];

    public function addRecipient(string $recipient, string $name = ''): MailerInterface
    {
        $this->to[] = $recipient;

        return $this;
    }

    public function addCc(string $cc, string $name = ''): MailerInterface
    {
        if ($name) {
            $this->headers[] = 'Cc:' . $name . ' <' . $cc . '>';

            return $this;
        }

        $this->headers[] = 'Cc:' . $cc;

        return $this;
    }

    public function addBcc(string $bcc, string $name = ''): MailerInterface
    {
        if ($name) {
            $this->headers[] = 'Bcc:' . $name . ' <' . $bcc . '>';

            return $this;
        }

        $this->headers[] = 'Bcc:' . $bcc;

        return $this;
    }

    public function addReplyTo(string $replyTo, string $name = ''): MailerInterface
    {
        if ($name) {
            $this->headers[] = 'Reply-To:' . $name . ' <' . $replyTo . '>';

            return $this;
        }

        $this->headers[] = 'Reply-To:' . $replyTo;

        return $this;
    }

    public function setFrom(string $from, string $name = ''): MailerInterface
    {
        if ($name) {
            $this->headers[] = 'From:' . $name . ' <' . $from . '>';

            return $this;
        }

        $this->headers[] = 'From:' . $from;

        return $this;
    }

    public function setSubject(string $subject): MailerInterface
    {
        $this->subject = $subject;

        return $this;
    }

    public function setBody(string $body): MailerInterface
    {
        $this->body = $body;

        return $this;
    }

    public function isHtml(bool $isHtml): MailerInterface
    {
        $this->html = $isHtml;

        return $this;
    }

    public function addAttachment(string $filePath): MailerInterface
    {
        if (!Str::startsWith($filePath, '~') && !Path::isAbsolute($filePath)) {
            $filePath = "~/$filePath";
        }

        if (File::exists($filePath)) {
            $this->attachments[] = Path::resolve($filePath);
        }

        return $this;
    }

    public function send(): bool
    {
        if ($this->html) {
            add_filter('wp_mail_content_type', [
                $this,
                'setEmailContentType'
            ]);
        }

        $result = wp_mail($this->to, $this->subject, $this->body, $this->headers, $this->attachments);

        if ($this->html) {
            remove_filter('wp_mail_content_type', [
                $this,
                'setEmailContentType'
            ]);
        }

        return $result;
    }

    public function setEmailContentType(): string
    {
        return 'text/html';
    }
}
