<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Access\Rule;

use YOOtheme\Arr;
use YOOtheme\Config;
use YOOtheme\Str;
use ZOOlanders\YOOessentials\Access\AccessRule;

/**
 * Code adapted from Regular Labs Library version 20.9.11663
 *
 * @author Peter van Westen
 * @copyright Copyright © 2020 Regular Labs All Rights Reserved
 * @license http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 */
class UrlRule implements AccessRule
{
    public function group() : string
    {
        return 'site';
    }

    public function name() : string
    {
        return 'Url';
    }

    public function namespace() : string
    {
        return 'yooessentials_access_url';
    }

    public function description() : string
    {
        return 'Validates if the current url matches the pattern.';
    }

    /**
     * @var Config
     */
    private $config;

    public function __construct(Config $config)
    {
        $this->config = $config;
    }

    public function resolve($props, $node) : bool
    {
        if (!isset($props->urls)) {
            return true;
        }

        $regex = $props->regex ?? false;
        $selection = explode(',', str_replace([' ', "\r", "\n"], ['', '', ','], $props->urls));

        return Arr::some($this->getUrls(), function ($url) use ($selection, $regex) {
            return Arr::some($selection, function ($s) use ($url, $regex) {
                if ($regex) {
                    $url_part = str_replace(['#', '&amp;'], ['\#', '(&amp;|&)'], $s);

                    $s = "#{$url_part}#si";

                    if (@preg_match("{$s}u", $url) || @preg_match($s, $url)) {
                        return true;
                    }
                }

                if (Str::contains($url, $s)) {
                    return true;
                }
            });
        });
    }

    public function fields() : array
    {
        return [
            "{$this->namespace()}.urls" => [
                'type' => 'textarea',
                'description' => 'A list of part URLs to match the current site url. Separate the entries with a comma and/or new line.',
                'attrs' => [
                    'rows' => 4,
                    'placeholder' => "localhost\nmy/site/section"
                ]
            ],
            "{$this->namespace()}.regex" => [
                'text' => 'Use Regular Expressions',
                'type' => 'checkbox'
            ]
        ];
    }

    protected function getUrls() : array
    {
        static $urls = [];

        if (empty($urls)) {
            $url = $this->config->get('req.href');

            $urls = [
                html_entity_decode(urldecode($url), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                urldecode($url),
                html_entity_decode($url, ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                $url,
            ];

            $urls = array_unique($urls);
        }

        return $urls;
    }
}
