<?php
/**
 * @package   Essentials YOOtheme Pro 1.2.7
 * @author    ZOOlanders https://www.zoolanders.com
 * @copyright Copyright (C) Joolanders, SL
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace ZOOlanders\YOOessentials\Access\Rule;

use DateTime;

class SeasonRule extends TimeRule
{
    public function name() : string
    {
        return 'Season';
    }

    public function namespace() : string
    {
        return 'yooessentials_access_season';
    }

    public function description() : string
    {
        return 'Validates if the current meteorological season matches the selection.';
    }

    public function resolve($props, $node) : bool
    {
        if (!isset($props->seasons)) {
            return true;
        }

        $now = new DateTime('now', $this->tz);
        $season = $this->getSeason($now, $props->hemisphere ?? false);

        return in_array($season, (array) $props->seasons);
    }

    public function fields() : array
    {
        return [
            "{$this->namespace()}.seasons" => [
                'type' => 'select',
                'description' => 'The seasons that the current date must match. Timezone from Site configuration is automatically applied. Use the shift or ctrl/cmd key to select multiple entries.',
                'attrs' => [
                    'multiple' => true
                ],
                'options' => [
                    'Winter' => 'winter',
                    'Spring' => 'spring',
                    'Summer' => 'summer',
                    'Autumn' => 'fall'
                ],
            ],
            "{$this->namespace()}.hemisphere" => [
                'type' => 'select',
                'options' => [
                    'Northern' => '',
                    'Southern' => 'southern',
                    'Australia' => 'australia'
                ],
                'description' => 'The Hemisphere from which to calculate the current season.'
            ]
        ];
    }

    /**
     * Code extracted from Regular Labs Library version 20.9.11663
     *
     * @author          Peter van Westen
     * @copyright       Copyright © 2020 Regular Labs All Rights Reserved
     * @license         http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
     */
    private function getSeason(&$d, $hemisphere = 'northern')
    {
        // Set $date to today
        $date = strtotime($d->format('Y-m-d H:i:s'));

        // Get year of date specified
        $date_year = $d->format('Y'); // Four digit representation for the year

        // Specify the season names
        $season_names = ['winter', 'spring', 'summer', 'fall'];

        // Declare season date ranges
        switch (strtolower($hemisphere)) {
            case 'southern':
                if (
                    $date < strtotime($date_year . '-03-21')
                    || $date >= strtotime($date_year . '-12-21')
                ) {
                    return $season_names[2]; // Must be in Summer
                }

                if ($date >= strtotime($date_year . '-09-23')) {
                    return $season_names[1]; // Must be in Spring
                }

                if ($date >= strtotime($date_year . '-06-21')) {
                    return $season_names[0]; // Must be in Winter
                }

                if ($date >= strtotime($date_year . '-03-21')) {
                    return $season_names[3]; // Must be in Fall
                }

                break;
            case 'australia':
                if (
                    $date < strtotime($date_year . '-03-01')
                    || $date >= strtotime($date_year . '-12-01')
                ) {
                    return $season_names[2]; // Must be in Summer
                }

                if ($date >= strtotime($date_year . '-09-01')) {
                    return $season_names[1]; // Must be in Spring
                }

                if ($date >= strtotime($date_year . '-06-01')) {
                    return $season_names[0]; // Must be in Winter
                }

                if ($date >= strtotime($date_year . '-03-01')) {
                    return $season_names[3]; // Must be in Fall
                }

                break;
            default: // northern
                if (
                    $date < strtotime($date_year . '-03-21')
                    || $date >= strtotime($date_year . '-12-21')
                ) {
                    return $season_names[0]; // Must be in Winter
                }

                if ($date >= strtotime($date_year . '-09-23')) {
                    return $season_names[3]; // Must be in Fall
                }

                if ($date >= strtotime($date_year . '-06-21')) {
                    return $season_names[2]; // Must be in Summer
                }

                if ($date >= strtotime($date_year . '-03-21')) {
                    return $season_names[1]; // Must be in Spring
                }

                break;
        }

        return 0;
    }
}
